//
//  PrinterBluetoothManager.h
//  PrinterBluetoothManager
//
//  Created by Mac on 1/11/21.
//

#import <Foundation/Foundation.h>
#import <CoreBluetooth/CoreBluetooth.h>

NS_ASSUME_NONNULL_BEGIN

@interface PrinterBluetoothManager : NSObject<CBCentralManagerDelegate,CBPeripheralDelegate>

//单个实例
//Single instance
+(instancetype) printerBluetoothManagerInstance;

//读取数据的闭包
//Closure for reading data
typedef void (^ReadNSData)(NSData * _Nullable, NSString *);

//读取打印设备数据回调
//Read print device data callback
@property (strong, nonatomic) ReadNSData _Nullable readNSData;

//扫描到设备的回调闭包
//The callback closure for scanning to the device
- (void)blePeripheralFound:(void (^)(CBPeripheral *peripheral, NSNumber *rssi))block;

//连接设备成功的回调闭包
//The callback closure for the successful connection of the device
- (void)blePeripheralConnected:(void (^)(CBCentralManager *central,CBPeripheral *peripheral))block;

//设备断开的回调闭包
//Callback closure for device disconnection
- (void)blePeripheralDisonnected:(void (^)(CBCentralManager *central,CBPeripheral *peripheral))block;

/**
 * 开始扫描打印机设备
 * Start scanning for printer devices
 */
-(void)beginScan;

/**
 * 停止扫描打印机设备
 * Stop scanning for printer devices
 */
-(void)stopScan;

/**
 * 连接打印机蓝牙
 * Connect printer Bluetooth
 */
-(void)connect:(CBPeripheral *)peripheral;


/**
 * 连接上一次连接的蓝牙，默认不连接
 * Connect to the last connected Bluetooth, not connected by default
 */
-(void)setIsOpenDefaultConnect:(BOOL)defaultConnect;

/**
 *发送数据给打印机
 *Send data to the printer
 */
-(BOOL)writeNSData:(NSData *)nsData;

/**
 * 发送数据给打印机并通过block 回调读取结果
 * Send data to the printer and read the result through block callback
 */
-(void)writeAndRead:(NSData *)nsData readNSData:(ReadNSData) readNSData;


//是否已经连接打印机
//Is the printer connected
-(BOOL)isConnected;

/**
 * 得到当前连接的蓝牙名称，若有，返回名称，否则返回NULL
 * Get the currently connected Bluetooth name, if there is, return the name, otherwise return NULL
 */
-(NSString *)getCurrentName;


//获取蓝牙地址
//Get Bluetooth address
-(NSString *)getMacAddress;


@end

NS_ASSUME_NONNULL_END

